// cpu.c: Signetics 2650 CPU emulation module (platform-independent)

#ifdef AMIGA
    #include "lint.h"
    #include "amiga.h"
#endif
#ifdef WIN32
    #include "ibm.h"
#endif

#include <stdio.h>             // for printf()
#include <string.h>            // for strcpy()

#include "aa.h"

#define gosub         pushras(); branch();
#define BRANCHCODE    (opcode & 3)
#define CCFIELD       ((psl & PSL_CC) >> 6)
#define ONE_BYTE      iar = (iar + 1) & PMSK;
#define TWO_BYTES     iar = (iar + 2) & PMSK;
#define THREE_BYTES   iar = (iar + 3) & PMSK;
#define SIX_CYCLES    cycles += 6;
#define NINE_CYCLES   cycles += 9;
#define TWELVE_CYCLES cycles += 12;
#define OPERAND       (memory[page + iar + 1]) // doesn't wrap properly
#define GET_RR        rr = (opcode & 3); if (rr && (psl & PSL_RS)) rr += 3;

/* When tracing, we actually can call REL_EA() etc. twice. addressstring
is appended to both times. addresstring would then be wrong and too long,
but it is actually used between the first and second calls (ie. it has
only been called once at that point), and the addressstring is long enough
that a buffer overrun cannot occur. */

EXPORT UBYTE table_sz[256] =
{   1, 1, 1, 1, 2, 2, 2, 2, 2, 2, 2, 2, 3, 3, 3, 3, //   0.. 15 (LOD)
    1, 1 ,1, 1, 1, 1, 1, 1, 2, 2, 2, 2, 3, 3, 3, 3, //  16.. 31 (SPSx/RETC/BCTR/BCTA)
    1, 1, 1, 1, 2, 2, 2, 2, 2, 2, 2, 2, 3, 3, 3, 3, //  32.. 47 (EOR)
    1, 1, 1, 1, 1, 1, 1, 1, 2, 2, 2, 2, 3, 3, 3, 3, //  48.. 63 (REDC/RETE/BSTR/BSTA)
    1, 1, 1, 1, 2, 2, 2, 2, 2, 2, 2, 2, 3, 3, 3, 3, //  64.. 79 (HALT/AND)
    1, 1, 1, 1, 2, 2, 2, 2, 2, 2, 2, 2, 3, 3, 3, 3, //  80.. 95 (RRR/REDE/BRNR/BRNA)
    1, 1, 1, 1, 2, 2, 2, 2, 2, 2, 2, 2, 3, 3, 3, 3, //  96..111 (IOR)
    1, 1, 1, 1, 2, 2, 2, 2, 2, 2, 2, 2, 3, 3, 3, 3, // 112..127 (REDD/CPSx/PPSx/BSNR/BSNA)
    1, 1, 1, 1, 2, 2, 2, 2, 2, 2, 2, 2, 3, 3, 3, 3, // 128..143 (ADD)
    1, 1 ,1, 1, 1, 1, 1, 1, 2, 2, 2, 2, 3, 3, 3, 3, // 144..159 (LPSx/DAR/BCFR/ZBRR/BCFA/BXA)
    1, 1, 1, 1, 2, 2, 2, 2, 2, 2, 2, 2, 3, 3, 3, 3, // 160..175 (SUB)
    1, 1, 1, 1, 2, 2, 1, 1, 2, 2, 2, 2, 3, 3, 3, 3, // 176..191 (WRTC/TPSx/BSFR/ZBSR/BSFA/BSXA)
    1, 1, 1, 1, 1, 1, 1, 1, 2, 2, 2, 2, 3, 3, 3, 3, // 192..207 (NOP/STR)
    1, 1, 1, 1, 2, 2, 2, 2, 2, 2, 2, 2, 3, 3, 3, 3, // 208..223 (RRL/WRTE/BIRR/BIRA)
    1, 1, 1, 1, 2, 2, 2, 2, 2, 2, 2, 2, 3, 3, 3, 3, // 224..239 (COM)
    1, 1, 1, 1, 2, 2, 2, 2, 2, 2, 2, 2, 3, 3, 3, 3  // 240..255 (WRTD/TMI/BDRR/BDRA)
};

MODULE int table_clockperiods[256] =
{   6, 6, 6, 6, 6, 6, 6, 6, 9, 9, 9, 9, 12, 12, 12, 12, //   0..15  (LOD)
    6, 6, 6, 6, 9, 9, 9, 9, 9, 9, 9, 9,  9,  9,  9,  9, //  16..31
    6, 6, 6, 6, 6, 6, 6, 6, 9, 9, 9, 9, 12, 12, 12, 12, //  32..47  (EOR)
    6, 6, 6, 6, 9, 9, 9, 9, 9, 9, 9, 9,  9,  9,  9,  9, //  48..63
    6, 6, 6, 6, 6, 6, 6, 6, 9, 9, 9, 9, 12, 12, 12, 12, //  64..79
    6, 6, 6, 6, 9, 9, 9, 9, 9, 9, 9, 9,  9,  9,  9,  9, //  80..95
    6, 6, 6, 6, 6, 6, 6, 6, 9, 9, 9, 9, 12, 12, 12, 12, //  96..111 (IOR)
    6, 6, 6, 6, 9, 9, 9, 9, 9, 9, 9, 9,  9,  9,  9,  9, // 112..127
    6, 6, 6, 6, 6, 6, 6, 6, 9, 9, 9, 9, 12, 12, 12, 12, // 128..143 (ADD)
    6, 6, 6, 6, 9, 9, 9, 9, 9, 9, 9, 9,  9,  9,  9,  9, // 144..159
    6, 6, 6, 6, 6, 6, 6, 6, 9, 9, 9, 9, 12, 12, 12, 12, // 160..175 (SUB)
    6, 6, 6, 6, 9, 9, 6, 6, 9, 9, 9, 9,  9,  9,  9,  9, // 176..191
    6, 6, 6, 6, 6, 6, 6, 6, 9, 9, 9, 9, 12, 12, 12, 12, // 192..207
    6, 6, 6, 6, 9, 9, 9, 9, 9, 9, 9, 9, 12, 12, 12, 12, // 208..223
    6, 6, 6, 6, 6, 6, 6, 6, 9, 9, 9, 9, 12, 12, 12, 12, // 224..239 (COM)
    6, 6, 6, 6, 9, 9, 9, 9, 9, 9, 9, 9,  9,  9,  9,  9  // 240..255
};
// Indirect addressing instructions require 15 or 18 cycles (6 more than usual).

/* condition code changes for a byte
   Apparently this table is wrong as regards the OVF bit.
   We do our own OVF updating afterwards to ensure it is correct. */
MODULE const UBYTE ccc[0x200] =
{   0x00,0x40,0x40,0x40,0x40,0x40,0x40,0x40,
    0x40,0x40,0x40,0x40,0x40,0x40,0x40,0x40,
    0x40,0x40,0x40,0x40,0x40,0x40,0x40,0x40,
    0x40,0x40,0x40,0x40,0x40,0x40,0x40,0x40,
    0x40,0x40,0x40,0x40,0x40,0x40,0x40,0x40,
    0x40,0x40,0x40,0x40,0x40,0x40,0x40,0x40,
    0x40,0x40,0x40,0x40,0x40,0x40,0x40,0x40,
    0x40,0x40,0x40,0x40,0x40,0x40,0x40,0x40,
    0x40,0x40,0x40,0x40,0x40,0x40,0x40,0x40,
    0x40,0x40,0x40,0x40,0x40,0x40,0x40,0x40,
    0x40,0x40,0x40,0x40,0x40,0x40,0x40,0x40,
    0x40,0x40,0x40,0x40,0x40,0x40,0x40,0x40,
    0x40,0x40,0x40,0x40,0x40,0x40,0x40,0x40,
    0x40,0x40,0x40,0x40,0x40,0x40,0x40,0x40,
    0x40,0x40,0x40,0x40,0x40,0x40,0x40,0x40,
    0x40,0x40,0x40,0x40,0x40,0x40,0x40,0x40,
    0x80,0x80,0x80,0x80,0x80,0x80,0x80,0x80,
    0x80,0x80,0x80,0x80,0x80,0x80,0x80,0x80,
    0x80,0x80,0x80,0x80,0x80,0x80,0x80,0x80,
    0x80,0x80,0x80,0x80,0x80,0x80,0x80,0x80,
    0x80,0x80,0x80,0x80,0x80,0x80,0x80,0x80,
    0x80,0x80,0x80,0x80,0x80,0x80,0x80,0x80,
    0x80,0x80,0x80,0x80,0x80,0x80,0x80,0x80,
    0x80,0x80,0x80,0x80,0x80,0x80,0x80,0x80,
    0x80,0x80,0x80,0x80,0x80,0x80,0x80,0x80,
    0x80,0x80,0x80,0x80,0x80,0x80,0x80,0x80,
    0x80,0x80,0x80,0x80,0x80,0x80,0x80,0x80,
    0x80,0x80,0x80,0x80,0x80,0x80,0x80,0x80,
    0x80,0x80,0x80,0x80,0x80,0x80,0x80,0x80,
    0x80,0x80,0x80,0x80,0x80,0x80,0x80,0x80,
    0x80,0x80,0x80,0x80,0x80,0x80,0x80,0x80,
    0x80,0x80,0x80,0x80,0x80,0x80,0x80,0x80,
    0x04,0x44,0x44,0x44,0x44,0x44,0x44,0x44,
    0x44,0x44,0x44,0x44,0x44,0x44,0x44,0x44,
    0x44,0x44,0x44,0x44,0x44,0x44,0x44,0x44,
    0x44,0x44,0x44,0x44,0x44,0x44,0x44,0x44,
    0x44,0x44,0x44,0x44,0x44,0x44,0x44,0x44,
    0x44,0x44,0x44,0x44,0x44,0x44,0x44,0x44,
    0x44,0x44,0x44,0x44,0x44,0x44,0x44,0x44,
    0x44,0x44,0x44,0x44,0x44,0x44,0x44,0x44,
    0x44,0x44,0x44,0x44,0x44,0x44,0x44,0x44,
    0x44,0x44,0x44,0x44,0x44,0x44,0x44,0x44,
    0x44,0x44,0x44,0x44,0x44,0x44,0x44,0x44,
    0x44,0x44,0x44,0x44,0x44,0x44,0x44,0x44,
    0x44,0x44,0x44,0x44,0x44,0x44,0x44,0x44,
    0x44,0x44,0x44,0x44,0x44,0x44,0x44,0x44,
    0x44,0x44,0x44,0x44,0x44,0x44,0x44,0x44,
    0x44,0x44,0x44,0x44,0x44,0x44,0x44,0x44,
    0x84,0x84,0x84,0x84,0x84,0x84,0x84,0x84,
    0x84,0x84,0x84,0x84,0x84,0x84,0x84,0x84,
    0x84,0x84,0x84,0x84,0x84,0x84,0x84,0x84,
    0x84,0x84,0x84,0x84,0x84,0x84,0x84,0x84,
    0x84,0x84,0x84,0x84,0x84,0x84,0x84,0x84,
    0x84,0x84,0x84,0x84,0x84,0x84,0x84,0x84,
    0x84,0x84,0x84,0x84,0x84,0x84,0x84,0x84,
    0x84,0x84,0x84,0x84,0x84,0x84,0x84,0x84,
    0x84,0x84,0x84,0x84,0x84,0x84,0x84,0x84,
    0x84,0x84,0x84,0x84,0x84,0x84,0x84,0x84,
    0x84,0x84,0x84,0x84,0x84,0x84,0x84,0x84,
    0x84,0x84,0x84,0x84,0x84,0x84,0x84,0x84,
    0x84,0x84,0x84,0x84,0x84,0x84,0x84,0x84,
    0x84,0x84,0x84,0x84,0x84,0x84,0x84,0x84,
    0x84,0x84,0x84,0x84,0x84,0x84,0x84,0x84,
    0x84,0x84,0x84,0x84,0x84,0x84,0x84,0x84
};

/***************************************************************
 * handy table to build PC relative offsets
 * from HR (holding register)
 ***************************************************************/
MODULE const int S2650_relative[0x100] =
{     0,  1,  2,  3,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14, 15,
     16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31,
     32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47,
     48, 49, 50, 51, 52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 62, 63,
    -64,-63,-62,-61,-60,-59,-58,-57,-56,-55,-54,-53,-52,-51,-50,-49,
    -48,-47,-46,-45,-44,-43,-42,-41,-40,-39,-38,-37,-36,-35,-34,-33,
    -32,-31,-30,-29,-28,-27,-26,-25,-24,-23,-22,-21,-20,-19,-18,-17,
    -16,-15,-14,-13,-12,-11,-10, -9, -8, -7, -6, -5, -4, -3, -2, -1,
      0,  1,  2,  3,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14, 15,
     16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31,
     32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47,
     48, 49, 50, 51, 52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 62, 63,
    -64,-63,-62,-61,-60,-59,-58,-57,-56,-55,-54,-53,-52,-51,-50,-49,
    -48,-47,-46,-45,-44,-43,-42,-41,-40,-39,-38,-37,-36,-35,-34,-33,
    -32,-31,-30,-29,-28,-27,-26,-25,-24,-23,-22,-21,-20,-19,-18,-17,
    -16,-15,-14,-13,-12,-11,-10, -9, -8, -7, -6, -5, -4, -3, -2, -1
};

EXPORT char ccstring[4][2 + 1] =
{   "eq",
    "gt",
    "lt",
    "un"
};

// IMPORTED VARIABLES-----------------------------------------------------

IMPORT int    bp,
              machine;
IMPORT FLAG   consoleopen,
              loginstructions,
              trace;
IMPORT TEXT   addrstr[13 + 1 + 1],
              friendly[21 + 1];
IMPORT ULONG  frames,
              paused;

// imported function pointers
IMPORT UBYTE (* cpuread)(int address);
IMPORT void (* cpuwrite)(int address, UBYTE data);

// MODULE VARIABLES-------------------------------------------------------

MODULE ULONG  ea;
MODULE SBYTE  ovf;
MODULE SWORD  res;
MODULE UBYTE  operand,
              before,
              c;
MODULE char   regstring[7][2 + 1],
              rasstring[8][6 + 1],
              iarstring[6 + 1],
              psubits[3 + 1],
              pslbits[6 + 1],
              byte1[2 + 1],
              byte2[2 + 1],
              byte3[2 + 1],
              v[2 + 1],
              reg[1 + 1] = "#", // does not include the 'r'
              mn[80 + 1],
              addressstring[9 + 1]; // "*$1234,r3+" is largest possible
MODULE int    addr;

// EXPORTED VARIABLES-----------------------------------------------------

EXPORT FLAG  interrupt,
             slow        = FALSE,
             step;
EXPORT UBYTE psu,
             psl,
             memory[32768],
             opcode,
             rr, // register (0..6)
             r[7]; // register is a reserved word
EXPORT UWORD iar,
             page, // page select register
             ras[8];
EXPORT ULONG cycles,
             elapsed;
EXPORT int   overcalc, // must be signed!
             slice;

MODULE __inline void set_cc(UBYTE n);
MODULE __inline void pushras(void);
MODULE __inline void pullras(void);
MODULE __inline void branch(void);
MODULE void compare(UBYTE first, UBYTE second);
MODULE UBYTE add(UBYTE dest, UBYTE source);
MODULE UBYTE subtract(UBYTE dest, UBYTE source);
MODULE void ABS_EA(void);
MODULE void BRA_EA(void);
MODULE void REL_EA(void);
MODULE void ZERO_EA(void);

// these are only needed for debugging
MODULE __inline void tracecpu(ULONG cycles);
MODULE void ABS_EA_TRACE(void);
MODULE void REL_EA_TRACE(void);
MODULE void BRA_EA_TRACE(void);
MODULE void ZERO_EA_TRACE(void);

EXPORT void cpu(void)
{   if (slice <= 0) // this can happen! remember some instructions take up to 18 cycles!
    {   overcalc = -slice;
        return;
    }

    if (interrupt)
    {   if (trace)
        {   OPENCONSOLE;
            printf("Generated interrupt.\n\n");
            REACTIVATE;
        }
        interrupt = FALSE;

        psu |= PSU_II; // set II bit (PSU |= %00100000)
        ras[psu & PSU_SP] = (SLONG) page + iar; // wraparound issues?
        // assert(machine != ARCADIA);
        ea = 0x0003;

        // increment SP
        if ((psu & PSU_SP) == 7) // overflow
        {   psu &= ~(PSU_SP); // set SP to 0
        } else
        {   psu++;
        }
        branch();
        NINE_CYCLES;
    }
/* Interrupt timing is late. Interrupts should presumably occur (ie. the
   branch should actually happen) before the next executed instruction
   after the interrupt flag has been set. Our granularity is not that
   fine. */

    do
    {   if (iar == bp && !step)
        {   getfriendly(bp);
            OPENCONSOLE;
            printf("Hit code breakpoint at %s!\n\n", friendly);
            REACTIVATE;
            runto();
        }

        /* Opcodes and operands should probably be fetched through
           cpuread(), because of mirroring, read/write status, etc.
           This would be slow, and would cause problems with
           watchpoints and read-once addresses. */

        opcode = memory[page + iar];

        if (trace)
        {   tracecpu(elapsed + cycles);
        }

        switch(opcode)
        {
        case 0:                                                 // LODZ r0
            if (loginstructions)
            {   OPENCONSOLE;
                printf("Instruction at $%lX executed indeterminate opcode $00 (LODZ r0)!\n\n", iar);
                REACTIVATE;
            }
            // hopefully this is the same behaviour as on a genuine console
            set_cc(r[0]);
            ONE_BYTE;
            SIX_CYCLES;
        acase 1:                                                // LODZ r1
        case 2:                                                 // LODZ r2
        case 3:                                                 // LODZ r3
            GET_RR;
            r[0] = r[rr];
            set_cc(r[0]);
            ONE_BYTE;
            SIX_CYCLES;
        acase 4:                                                // LODI,r0
            r[0] = OPERAND;
            set_cc(r[0]);
            TWO_BYTES;
            SIX_CYCLES;
        acase 5:                                                // LODI,r1
        case 6:                                                 // LODI,r2
        case 7:                                                 // LODI,r3
            GET_RR;
            r[rr] = OPERAND;
            set_cc(r[rr]);
            TWO_BYTES;
            SIX_CYCLES;
        acase 8:                                                // LODR,r0
            REL_EA();
            r[0] = cpuread(ea);
            set_cc(r[0]);
            TWO_BYTES;
            NINE_CYCLES;
        acase 9:                                                // LODR,r1
        case 10:                                                // LODR,r2
        case 11:                                                // LODR,r3
            REL_EA();
            GET_RR;
            r[rr] = cpuread(ea);
            set_cc(r[rr]);
            TWO_BYTES;
            NINE_CYCLES;
        acase 12:                                               // LODA,r0
        case 13:                                                // LODA,r1
        case 14:                                                // LODA,r2
        case 15:                                                // LODA,r3
            ABS_EA(); // calls GET_RR for us
            r[rr] = cpuread(ea);
            set_cc(r[rr]);
            THREE_BYTES;
            TWELVE_CYCLES;
        acase 18:                                               // SPSU
            r[0] = psu;
            set_cc(psu);
            ONE_BYTE;
            SIX_CYCLES;
        acase 19:                                               // SPSL
            r[0] = psl;
            set_cc(psl);
            ONE_BYTE;
            SIX_CYCLES;
        acase 20:                                               // RETC,eq
        case 21:                                                // RETC,gt
        case 22:                                                // RETC,lt
            if (BRANCHCODE == CCFIELD)
            {   pullras();
            } else
            {   ONE_BYTE;
            }
            NINE_CYCLES;
        acase 23:                                               // RETC,un
            pullras();
            NINE_CYCLES;
        acase 24:                                               // BCTR,eq
        case 25:                                                // BCTR,gt
        case 26:                                                // BCTR,lt
            if (BRANCHCODE == CCFIELD)
            {   REL_EA();
                branch();
            } else
            {   TWO_BYTES;
            }
            NINE_CYCLES;
        acase 27:                                               // BCTR,un
            REL_EA();
            branch();
            NINE_CYCLES;
        acase 28:                                               // BCTA,eq
        case 29:                                                // BCTA,gt
        case 30:                                                // BCTA,lt
            if (BRANCHCODE == CCFIELD)
            {   BRA_EA();
                branch();
            } else
            {   THREE_BYTES;
            }
            NINE_CYCLES;
        acase 31:                                               // BCTA,un
            BRA_EA();
            branch();
            NINE_CYCLES;
        acase 32:                                               // EORZ r0
            r[0] = 0;
            psl &= ~(PSL_CC); // clear CC bits of PSL (ie. CC = "eq")
            ONE_BYTE;
            SIX_CYCLES;
        acase 33:                                               // EORZ r1
        case 34:                                                // EORZ r2
        case 35:                                                // EORZ r3
            GET_RR;
            r[0] ^= r[rr];
            set_cc(r[0]);
            ONE_BYTE;
            SIX_CYCLES;
        acase 36:                                               // EORI,r0
            r[0] ^= OPERAND;
            set_cc(r[0]);
            TWO_BYTES;
            SIX_CYCLES;
        acase 37:                                               // EORI,r1
        case 38:                                                // EORI,r2
        case 39:                                                // EORI,r3
            GET_RR;
            r[rr] ^= OPERAND;
            set_cc(r[rr]);
            TWO_BYTES;
            SIX_CYCLES;
        acase 40:                                               // EORR,r0
            REL_EA();
            r[0] ^= cpuread(ea);
            set_cc(r[0]);
            TWO_BYTES;
            NINE_CYCLES;
        acase 41:                                               // EORR,r1
        case 42:                                                // EORR,r2
        case 43:                                                // EORR,r3
            REL_EA();
            GET_RR;
            r[rr] ^= cpuread(ea);
            set_cc(r[rr]);
            TWO_BYTES;
            NINE_CYCLES;
        acase 44:                                               // EORA,r0
        case 45:                                                // EORA,r1
        case 46:                                                // EORA,r2
        case 47:                                                // EORA,r3
            ABS_EA(); // calls GET_RR for us
            r[rr] ^= cpuread(ea);
            set_cc(r[rr]);
            THREE_BYTES;
            TWELVE_CYCLES;
        acase 52:                                               // RETE,eq
        case 53:                                                // RETE,gt
        case 54:                                                // RETE,lt
            if (loginstructions && machine == ARCADIA)
            {   OPENCONSOLE;
                printf("Instruction at $%lX executed strange opcode $%lX (RETE,%s)!\n\n", iar, opcode, ccstring[BRANCHCODE]);
                REACTIVATE;
            }
            if (BRANCHCODE == CCFIELD)
            {   psu &= ~(PSU_II); // clear II bit (PSU &= %11011111)
                pullras();
            } else
            {   ONE_BYTE;
            }
            NINE_CYCLES;
        acase 55:                                               // RETE,un
            if (loginstructions && machine == ARCADIA)
            {   OPENCONSOLE;
                printf("Instruction at $%lX executed strange opcode $37 (RETE,un)!\n\n", iar);
                REACTIVATE;
            }
            psu &= ~(PSU_II); // clear II bit (PSU &= %11011111)
            pullras();
            NINE_CYCLES;
        acase 56:                                               // BSTR,eq
        case 57:                                                // BSTR,gt
        case 58:                                                // BSTR,lt
            if (BRANCHCODE == CCFIELD)
            {   REL_EA();
                gosub;
            } else
            {   TWO_BYTES;
            }
            NINE_CYCLES;
        acase 59:                                               // BSTR,un
            REL_EA();
            gosub;
            NINE_CYCLES;
        acase 60:                                               // BSTA,eq
        case 61:                                                // BSTA,gt
        case 62:                                                // BSTA,lt
            if (BRANCHCODE == CCFIELD)
            {   BRA_EA();
                gosub;
            } else
            {   THREE_BYTES;
            }
            NINE_CYCLES;
        acase 63:                                               // BSTA,un
            BRA_EA();
            gosub;
            NINE_CYCLES;
        acase 64:                                               // HALT
            // strictly speaking, we should halt the emulation
            // entirely, instead of just repeatedly executing
            // the HALT instruction. eg. interrupts will continue to
            // occur, is this allowable?
            // ZERO_BYTES;
            SIX_CYCLES;
        acase 65:                                               // ANDZ r1
        case 66:                                                // ANDZ r2
        case 67:                                                // ANDZ r3
            GET_RR;
            r[0] &= r[rr];
            set_cc(r[0]);
            ONE_BYTE;
            SIX_CYCLES;
        acase 68:                                               // ANDI,r0
            r[0] &= OPERAND;
            set_cc(r[0]);
            TWO_BYTES;
            SIX_CYCLES;
        acase 69:                                               // ANDI,r1
        case 70:                                                // ANDI,r2
        case 71:                                                // ANDI,r3
            GET_RR;
            r[rr] &= OPERAND;
            set_cc(r[rr]);
            TWO_BYTES;
            SIX_CYCLES;
        acase 72:                                               // ANDR,r0
            REL_EA();
            r[0] &= cpuread(ea);
            set_cc(r[0]);
            TWO_BYTES;
            NINE_CYCLES;
        acase 73:                                               // ANDR,r1
        case 74:                                                // ANDR,r2
        case 75:                                                // ANDR,r3
            REL_EA();
            GET_RR;
            r[rr] &= cpuread(ea);
            set_cc(r[rr]);
            TWO_BYTES;
            NINE_CYCLES;
        acase 76:                                               // ANDA,r0
        case 77:                                                // ANDA,r1
        case 78:                                                // ANDA,r2
        case 79:                                                // ANDA,r3
            ABS_EA(); // calls GET_RR for us
            r[rr] &= cpuread(ea);
            set_cc(r[rr]);
            THREE_BYTES;
            TWELVE_CYCLES;
        acase 80:                                               // RRR,r0
        case 81:                                                // RRR,r1
        case 82:                                                // RRR,r2
        case 83:                                                // RRR,r3
            GET_RR;
            before = r[rr];
            if (psl & PSL_WC)
            {   c = psl & PSL_C;
                psl &= ~(PSL_C + PSL_IDC);
                r[rr] = (before >> 1) | (c << 7);
                psl |= (before & PSL_C) + (r[rr] & PSL_IDC);
            } else
            {   r[rr] = (before >> 1) | (before << 7);
            }
            psl = (psl & ~(PSL_OVF | PSL_CC)) | ccc[r[rr] + (((r[rr] ^ before) << 1) & 256)];
            if ((before & 0x80) != (r[rr] & 0x80))
            {   psl |= PSL_OVF;
            } else
            {   psl &= ~(PSL_OVF);
            }
            // set_cc() not necessary
            ONE_BYTE;
            SIX_CYCLES;
        acase 88:                                               // BRNR,r0
            if (r[0])
            {   REL_EA();
                branch();
            } else
            {   TWO_BYTES;
            }
            NINE_CYCLES;
        acase 89:                                               // BRNR,r1
        case 90:                                                // BRNR,r2
        case 91:                                                // BRNR,r3
            GET_RR;
            if (r[rr])
            {   REL_EA();
                branch();
            } else
            {   TWO_BYTES;
            }
            NINE_CYCLES;
        acase 92:                                               // BRNA,r0
            if (r[0])
            {   BRA_EA();
                branch();
            } else
            {   THREE_BYTES;
            }
            NINE_CYCLES;
        acase 93:                                               // BRNA,r1
        case 94:                                                // BRNA,r2
        case 95:                                                // BRNA,r3
            GET_RR;
            if (r[rr])
            {   BRA_EA();
                branch();
            } else
            {   THREE_BYTES;
            }
            NINE_CYCLES;
        acase 96:                                               // IORZ r0
            // r[0] |= r[0]; is not needed
            set_cc(r[0]);
            ONE_BYTE;
            SIX_CYCLES;
        acase 97:                                               // IORZ r1
        case 98:                                                // IORZ r2
        case 99:                                                // IORZ r3
            GET_RR;
            r[0] |= r[rr];
            set_cc(r[0]);
            ONE_BYTE;
            SIX_CYCLES;
        acase 100:                                              // IORI,r0
            r[0] |= OPERAND;
            set_cc(r[0]);
            TWO_BYTES;
            SIX_CYCLES;
        acase 101:                                              // IORI,r1
        case 102:                                               // IORI,r2
        case 103:                                               // IORI,r3
            GET_RR;
            r[rr] |= OPERAND;
            set_cc(r[rr]);
            TWO_BYTES;
            SIX_CYCLES;
        acase 104:                                              // IORR,r0
            REL_EA();
            r[0] |= cpuread(ea);
            set_cc(r[0]);
            TWO_BYTES;
            NINE_CYCLES;
        acase 105:                                              // IORR,r1
        case 106:                                               // IORR,r2
        case 107:                                               // IORR,r3
            REL_EA();
            GET_RR;
            r[rr] |= cpuread(ea);
            set_cc(r[rr]);
            TWO_BYTES;
            NINE_CYCLES;
        acase 108:                                              // IORA,r0
        case 109:                                               // IORA,r1
        case 110:                                               // IORA,r2
        case 111:                                               // IORA,r3
            ABS_EA(); // calls GET_RR for us
            r[rr] |= cpuread(ea);
            set_cc(r[rr]);
            THREE_BYTES;
            TWELVE_CYCLES;
        acase 116:                                              // CPSU
            psu &= (~(OPERAND & PSU_WRITABLE));
            TWO_BYTES;
            NINE_CYCLES;
        acase 117:                                              // CPSL
            psl &= (~OPERAND);
            TWO_BYTES;
            NINE_CYCLES;
        acase 118:                                              // PPSU
            psu |= (OPERAND & PSU_WRITABLE);
            TWO_BYTES;
            NINE_CYCLES;
        acase 119:                                              // PPSL
            psl |= OPERAND;
            TWO_BYTES;
            NINE_CYCLES;
        acase 120:                                              // BSNR,r0
            if (r[0])
            {   REL_EA();
                gosub;
            } else
            {   TWO_BYTES;
            }
            NINE_CYCLES;
        acase 121:                                              // BSNR,r1
        case 122:                                               // BSNR,r2
        case 123:                                               // BSNR,r3
            GET_RR;
            if (r[rr])
            {   REL_EA();
                gosub;
            } else
            {   TWO_BYTES;
            }
            NINE_CYCLES;
        acase 124:                                              // BSNA,r0
            if (r[0])
            {   BRA_EA();
                gosub;
            } else
            {   THREE_BYTES;
            }
            NINE_CYCLES;
        acase 125:                                              // BSNA,r1
        case 126:                                               // BSNA,r2
        case 127:                                               // BSNA,r3
            GET_RR;
            if (r[rr])
            {   BRA_EA();
                gosub;
            } else
            {   THREE_BYTES;
            }
            NINE_CYCLES;
        acase 128:                                              // ADDZ r0
            r[0] = add(r[0], r[0]);
            ONE_BYTE;
            SIX_CYCLES;
        acase 129:                                              // ADDZ r1
        case 130:                                               // ADDZ r2
        case 131:                                               // ADDZ r3
            GET_RR;
            r[0] = add(r[0], r[rr]);
            ONE_BYTE;
            SIX_CYCLES;
        acase 132:                                              // ADDI,r0
            r[0] = add(r[0], OPERAND);
            TWO_BYTES;
            SIX_CYCLES;
        acase 133:                                              // ADDI,r1
        case 134:                                               // ADDI,r2
        case 135:                                               // ADDI,r3
            GET_RR;
            r[rr] = add(r[rr], OPERAND);
            TWO_BYTES;
            SIX_CYCLES;
        acase 136:                                              // ADDR,r0
            REL_EA();
            r[0] = add(r[0], cpuread(ea));
            TWO_BYTES;
            NINE_CYCLES;
        acase 137:                                              // ADDR,r1
        case 138:                                               // ADDR,r2
        case 139:                                               // ADDR,r3
            REL_EA();
            GET_RR;
            r[rr] = add(r[rr], cpuread(ea));
            TWO_BYTES;
            NINE_CYCLES;
        acase 140:                                              // ADDA,r0
        case 141:                                               // ADDA,r1
        case 142:                                               // ADDA,r2
        case 143:                                               // ADDA,r3
            ABS_EA(); // calls GET_RR for us
            r[rr] = add(r[rr], cpuread(ea));
            THREE_BYTES;
            TWELVE_CYCLES;
        acase 146:                                              // LPSU r0
            psu &= PSU_READONLY;
            psu |= (r[0] & PSU_WRITABLE);
            ONE_BYTE;
            SIX_CYCLES;
        acase 147:                                              // LPSL r0
            psl = r[0];
            ONE_BYTE;
            SIX_CYCLES;
        acase 148:                                              // DAR  r0
            if ((psl & PSL_C) == 0)
            {   // add ten to the tens digit (the high nybble)
                r[0] += 0xA0; // 160. overflow OK. %1010,0000
            }
            if ((psl & PSL_IDC) == 0)
            {   // OR the high nybble with the (low nybble + 10)
                r[0] = (r[0] & 0xF0) | ((r[0] + 0x0A) & 0x0F);
            }
            set_cc(r[0]);
            ONE_BYTE;
            NINE_CYCLES;
        acase 149:                                              // DAR  r1
        case 150:                                               // DAR  r2
        case 151:                                               // DAR  r3
            GET_RR;
            if ((psl & PSL_C) == 0)
            {   // add ten to the tens digit (the high nybble)
                r[rr] += 0xA0; // 160. overflow OK. %1010,0000
            }
            if ((psl & PSL_IDC) == 0)
            {   // OR the high nybble with the (low nybble + 10)
                r[rr] = (r[rr] & 0xF0) | ((r[rr] + 0x0A) & 0x0F);
            }
            set_cc(r[rr]);
            ONE_BYTE;
            NINE_CYCLES;
        acase 152:                                              // BCFR,eq
        case 153:                                               // BCFR,gt
        case 154:                                               // BCFR,lt
            if (BRANCHCODE != CCFIELD)
            {   REL_EA();
                branch();
            } else
            {   TWO_BYTES;
            }
            NINE_CYCLES;
        acase 155:                                              // ZBRR,un
            ZERO_EA();
            branch();
            NINE_CYCLES;
        acase 156:                                              // BCFA,eq
        case 157:                                               // BCFA,gt
        case 158:                                               // BCFA,lt
            if (BRANCHCODE != CCFIELD)
            {   BRA_EA();
                branch();
            } else
            {   THREE_BYTES;
            }
            NINE_CYCLES;
        acase 159:                                              // BXA,r3
            BRA_EA();
            if (psl & PSL_RS)
            {   ea = (ea + r[6]) & AMSK;
            } else
            {   ea = (ea + r[3]) & AMSK;
            }
            branch();
            NINE_CYCLES;
        acase 160:                                              // SUBZ r0
            r[0] = subtract(r[0], r[0]);
            ONE_BYTE;
            SIX_CYCLES;
        acase 161:                                              // SUBZ r1
        case 162:                                               // SUBZ r2
        case 163:                                               // SUBZ r3
            GET_RR;
            r[0] = subtract(r[0], r[rr]);
            ONE_BYTE;
            SIX_CYCLES;
        acase 164:                                              // SUBI,r0
            r[0] = subtract(r[0], OPERAND);
            TWO_BYTES;
            SIX_CYCLES;
        acase 165:                                              // SUBI,r1
        case 166:                                               // SUBI,r2
        case 167:                                               // SUBI,r3
            GET_RR;
            r[rr] = subtract(r[rr], OPERAND);
            TWO_BYTES;
            SIX_CYCLES;
        acase 168:                                              // SUBR,r0
            REL_EA();
            r[0] = subtract(r[0], cpuread(ea));
            TWO_BYTES;
            NINE_CYCLES;
        acase 169:                                              // SUBR,r1
        case 170:                                               // SUBR,r2
        case 171:                                               // SUBR,r3
            REL_EA();
            GET_RR;
            r[rr] = subtract(r[rr], cpuread(ea));
            TWO_BYTES;
            NINE_CYCLES;
        acase 172:                                              // SUBA,r0
        case 173:                                               // SUBA,r1
        case 174:                                               // SUBA,r2
        case 175:                                               // SUBA,r3
            ABS_EA(); // this calls GET_RR for us
            r[rr] = subtract(r[rr], cpuread(ea));
            THREE_BYTES;
            TWELVE_CYCLES;
        acase 180:                                              // TPSU
            psl &= 0x3F; // set CC to %00xxxxxx ("eq")
            if ((psu & OPERAND) < OPERAND)
            {   psl |= 0x80; // set CC to %10xxxxxx ("lt")
            }
            TWO_BYTES;
            NINE_CYCLES;
        acase 181:                                              // TPSL
            psl &= 0x3F; // set CC to %00xxxxxx ("eq")
            if ((psl & OPERAND) < OPERAND)
            {   psl |= 0x80; // set CC to %10xxxxxx ("lt")
            }
            TWO_BYTES;
            NINE_CYCLES;
        acase 184:                                              // BSFR,eq
        case 185:                                               // BSFR,gt
        case 186:                                               // BSFR,lt
            if (BRANCHCODE != CCFIELD)
            {   REL_EA();
                gosub;
            } else
            {   TWO_BYTES;
            }
            NINE_CYCLES;
        acase 187:                                              // ZBSR,un
            ZERO_EA();
            gosub;
            NINE_CYCLES;
        acase 188:                                              // BSFA,eq
        case 189:                                               // BSFA,gt
        case 190:                                               // BSFA,lt
            if (BRANCHCODE != CCFIELD)
            {   BRA_EA();
                gosub;
            } else
            {   THREE_BYTES;
            }
            NINE_CYCLES;
        acase 191:                                              // BSXA,r3
            BRA_EA();
            if (psl & PSL_RS)
            {   ea = (ea + r[6]) & AMSK;
            } else
            {   ea = (ea + r[3]) & AMSK;
            }
            gosub;
            NINE_CYCLES;
        acase 192:                                              // NOP
            ONE_BYTE;
            SIX_CYCLES;
        acase 193:                                              // STRZ r1
        case 194:                                               // STRZ r2
        case 195:                                               // STRZ r3
            GET_RR;
            r[rr] = r[0];
            set_cc(r[rr]);
            ONE_BYTE;
            SIX_CYCLES;
        acase 200:                                              // STRR,r0
            REL_EA();
            cpuwrite((int) ea, r[0]);
            TWO_BYTES;
            NINE_CYCLES;
        acase 201:                                              // STRR,r1
        case 202:                                               // STRR,r2
        case 203:                                               // STRR,r3
            REL_EA();
            GET_RR;
            cpuwrite((int) ea, r[rr]);
            TWO_BYTES;
            NINE_CYCLES;
        acase 204:                                              // STRA,r0
        case 205:                                               // STRA,r1
        case 206:                                               // STRA,r2
        case 207:                                               // STRA,r3
            ABS_EA(); // this calls GET_RR for us
            cpuwrite((int) ea, r[rr]);
            THREE_BYTES;
            TWELVE_CYCLES;
        acase 208:                                              // RRL,r0
        case 209:                                               // RRL,r1
        case 210:                                               // RRL,r2
        case 211:                                               // RRL,r3
            GET_RR;
            before = r[rr];
            if (psl & PSL_WC)
            {   c = psl & PSL_C;
                psl &= ~(PSL_C + PSL_IDC);
                r[rr] = (before << 1) | c;
                psl |= (before >> 7) + (r[rr] & PSL_IDC);
            } else
            {   r[rr] = (before << 1) | (before >> 7);
            }
            psl = (psl & ~(PSL_OVF | PSL_CC)) | ccc[r[rr] + (((r[rr] ^ before) << 1) & 256)];
            if ((before & 0x80) != (r[rr] & 0x80))
            {   psl |= PSL_OVF;
            } else
            {   psl &= ~(PSL_OVF);
            }
            // set_cc() not necessary
            ONE_BYTE;
            SIX_CYCLES;
        acase 216:                                              // BIRR,r0
            if (++r[0]) // overflow OK
            {   REL_EA();
                branch();
            } else
            {   TWO_BYTES;
            }
            NINE_CYCLES;
        acase 217:                                              // BIRR,r1
        case 218:                                               // BIRR,r2
        case 219:                                               // BIRR,r3
            GET_RR;
            if (++r[rr]) // overflow OK
            {   REL_EA();
                branch();
            } else
            {   TWO_BYTES;
            }
            NINE_CYCLES;
        acase 220:                                              // BIRA,r0
            if (++r[0]) // overflow OK
            {   BRA_EA();
                branch();
            } else
            {   THREE_BYTES;
            }
            TWELVE_CYCLES;
        acase 221:                                              // BIRA,r1
        case 222:                                               // BIRA,r2
        case 223:                                               // BIRA,r3
            GET_RR;
            if (++r[rr]) // overflow OK
            {   BRA_EA();
                branch();
            } else
            {   THREE_BYTES;
            }
            TWELVE_CYCLES;
        acase 224:                                              // COMZ r0
            psl &= ~(PSL_CC); // clear CC bits of PSL
            ONE_BYTE;
            SIX_CYCLES;
        acase 225:                                              // COMZ r1
        case 226:                                               // COMZ r2
        case 227:                                               // COMZ r3
            GET_RR;
            compare(r[0], r[rr]);
            ONE_BYTE;
            SIX_CYCLES;
        acase 228:                                              // COMI,r0
            compare(r[0], OPERAND);
            TWO_BYTES;
            SIX_CYCLES;
        acase 229:                                              // COMI,r1
        case 230:                                               // COMI,r2
        case 231:                                               // COMI,r3
            GET_RR;
            compare(r[rr], OPERAND);
            TWO_BYTES;
            SIX_CYCLES;
        acase 232:                                              // COMR,r0
            REL_EA();
            compare(r[0], cpuread(ea));
            TWO_BYTES;
            NINE_CYCLES;
        acase 233:
        case 234:
        case 235:
            REL_EA();
            GET_RR;
            compare(r[rr], cpuread(ea));
            TWO_BYTES;
            NINE_CYCLES;
        acase 236:                                              // COMA,r0
        case 237:                                               // COMA,r1
        case 238:                                               // COMA,r2
        case 239:                                               // COMA,r3
            ABS_EA(); // this calls GET_RR for us
            compare(r[rr], cpuread(ea));
            THREE_BYTES;
            TWELVE_CYCLES;
        acase 244:                                              // TMI,r0
            psl &= 0x3F; // set CC to %00 ("eq")
            if ((r[0] & OPERAND) < OPERAND)
            {   psl |= 0x80; // set CC to %10 ("lt")
            }
            TWO_BYTES;
            NINE_CYCLES;
        acase 245:                                              // TMI,r1
        case 246:                                               // TMI,r2
        case 247:                                               // TMI,r3
            GET_RR;
            psl &= 0x3F; // set CC to %00 ("eq")
            if ((r[rr] & OPERAND) < OPERAND)
            {   psl |= 0x80; // set CC to %10 ("lt")
            }
            TWO_BYTES;
            NINE_CYCLES;
        acase 248:                                              // BDRR,r0
            if (--r[0]) // underflow OK
            {   REL_EA();
                branch();
            } else
            {   TWO_BYTES;
            }
            NINE_CYCLES;
        acase 249:                                              // BDRR,r1
        case 250:                                               // BDRR,r2
        case 251:                                               // BDRR,r3
            GET_RR;
            if (--r[rr]) // underflow OK
            {   REL_EA();
                branch();
            } else
            {   TWO_BYTES;
            }
            NINE_CYCLES;
        acase 252:                                              // BDRA,r0
            if (--r[0]) // underflow OK
            {   BRA_EA();
                branch();
            } else
            {   THREE_BYTES;
            }
            NINE_CYCLES;
        acase 253:                                              // BDRA,r1
        case 254:                                               // BDRA,r2
        case 255:                                               // BDRA,r3
            GET_RR;
            if (--r[rr]) // underflow OK
            {   BRA_EA();
                branch();
            } else
            {   THREE_BYTES;
            }
            NINE_CYCLES;
        // useless opcodes -----------------------------------------------
        acase 48:                                               // REDC r0
        case 49:                                                // REDC r1
        case 50:                                                // REDC r2
        case 51:                                                // REDC r3
            GET_RR;
            if (loginstructions)
            {   reg[0] = '0' + rr;
                OPENCONSOLE;
                printf("Instruction at $%lX executed strange opcode $%lX (REDC r%s)!\n\n", iar, opcode, reg);
                REACTIVATE;
            }
            r[rr] = 0;
            set_cc(r[rr]);
            ONE_BYTE;
            SIX_CYCLES;
        acase 84:                                               // REDE,r0
        case 85:                                                // REDE,r1
        case 86:                                                // REDE,r2
        case 87:                                                // REDE,r3
            GET_RR;
            if (loginstructions)
            {   reg[0] = '0' + rr;
                OPENCONSOLE;
                printf("Instruction at $%lX executed strange opcode $%lX (REDE r%s)!\n\n", iar, opcode, reg);
                // we could also give the operand
                REACTIVATE;
            }
            r[rr] = 0;
            set_cc(r[rr]);
            TWO_BYTES;
            NINE_CYCLES;
        acase 112:                                              // REDD r0
        case 113:                                               // REDD r1
        case 114:                                               // REDD r2
        case 115:                                               // REDD r3
            GET_RR;
            if (loginstructions)
            {   reg[0] = '0' + rr;
                OPENCONSOLE;
                printf("Instruction at $%lX executed strange opcode $%lX (REDD r%s)!\n\n", iar, opcode, reg);
                REACTIVATE;
            }
            r[rr] = 0;
            set_cc(r[rr]);
            ONE_BYTE;
            SIX_CYCLES;
        acase 176:
        case 177:
        case 178:
        case 179:
            if (loginstructions)
            {   GET_RR;
                reg[0] = '0' + rr;
                OPENCONSOLE;
                printf("Instruction at $%lX executed strange opcode $%lX (WRTC r%s)!\n\n", iar, opcode, reg);
                REACTIVATE;
            }
            ONE_BYTE;
            SIX_CYCLES;
        acase 212:
        case 213:
        case 214:
        case 215:
            if (loginstructions)
            {   GET_RR;
                reg[0] = '0' + rr;
                OPENCONSOLE;
                printf("Instruction at $%lX executed strange opcode $%lX (WRTE r%s)!\n\n", iar, opcode, reg);
                // we could also give the operand
                REACTIVATE;
            }
            TWO_BYTES;
            NINE_CYCLES;
        acase 240:
        case 241:
        case 242:
        case 243:
            if (loginstructions)
            {   GET_RR;
                reg[0] = '0' + rr;
                OPENCONSOLE;
                printf("Instruction at $%lX executed strange opcode $%lX (WRTD r%s)!\n\n", iar, opcode, reg);
                REACTIVATE;
            }
            ONE_BYTE;
            SIX_CYCLES;
        acase 0x10:
        case 0x11:
        case 0x90:
        case 0x91:
        case 0xB6:
        case 0xB7:
        case 0xC4:
        case 0xC5:
        case 0xC6:
        case 0xC7:
            if (loginstructions)
            {   OPENCONSOLE;
                printf("Instruction at $%lX executed undefined opcode $%lX!\n\n", iar, opcode);
                REACTIVATE;
            }
            // rarely used "do nothing" opcodes
            // These are all currently defined as 1 byte and 6 cycles
            iar = (iar + table_sz[opcode]) & PMSK;
            cycles += table_clockperiods[opcode];
        adefault:
            // assert(0);
        break;
        }

// it would be good not to have to test trace twice per instruction :-(
        if (trace)
        {   OPENCONSOLE;
            printf("%s\n", mn);
            view_cpu();
            // REACTIVATE; not needed as printcpu() does it
#ifdef WIN32
            updatemonitor(); // #ifdef WIN32 is for speed on Amiga
#endif
            updatescreen();
        }
        if (step)
        {   if (!trace)
            {
#ifdef WIN32
                updatemonitor(); // #ifdef WIN32 is for speed on Amiga
#endif
                updatescreen();
            }
            step = FALSE;
            pause();

            while (paused)
            {   checkinput(); // some things are unsafe for the user to do here!
    }   }   }
    while (cycles < (ULONG) slice);

    overcalc = cycles - slice;
    elapsed += cycles;
    cycles = 0; // needed
}

// -----------------------------------------------------------------------

EXPORT void view_cpu(void)
{   int i;

    hex1(regstring[0], r[0]);
    hex1(regstring[1], r[1]);
    hex1(regstring[2], r[2]);
    hex1(regstring[3], r[3]);
    hex1(regstring[4], r[4]);
    hex1(regstring[5], r[5]);
    hex1(regstring[6], r[6]);
    if (psu & PSU_S  ) psubits[0] = 'S'; else psubits[0] = 's';
    if (psu & PSU_F  ) psubits[1] = 'F'; else psubits[1] = 'f';
    if (psu & PSU_II ) psubits[2] = 'I'; else psubits[2] = 'i';
    psubits[3] = 0;
    if (psl & PSL_IDC) pslbits[0] = 'D'; else pslbits[0] = 'd';
    if (psl & PSL_RS ) pslbits[1] = 'R'; else pslbits[1] = 'r';
    if (psl & PSL_WC ) pslbits[2] = 'W'; else pslbits[2] = 'w';
    if (psl & PSL_OVF) pslbits[3] = 'O'; else pslbits[3] = 'o';
    if (psl & PSL_COM) pslbits[4] = 'M'; else pslbits[4] = 'm';
    if (psl & PSL_C  ) pslbits[5] = 'C'; else pslbits[5] = 'c';
    pslbits[6] = 0;

    OPENCONSOLE;
    printf // 40 leading spaces
    (   "                                 R0: $%s R1: $%s R2: $%s R3: $%s PSU: %s\n",
         regstring[0],
         regstring[1],
         regstring[2],
         regstring[3],
         psubits
    );
    printf
    (   "                                 CC: %s  R4: $%s R5: $%s R6: $%s PSL: %s\n",
         ccstring[(psl & 0xC0) >> 6], // %11000000 -> %00000011
         regstring[4],
         regstring[5],
         regstring[6],
         pslbits
    );

    for (i = 0; i <= 7; i++)
    {   hex2(rasstring[i], (UWORD) ras[i]);
    }
    hex2(iarstring, (UWORD) (page + iar));
    printf // 40 leading spaces
    (   "                                 SP: %ld   $%s $%s $%s $%s IAR: $%s\n",
        (SLONG) (psu & PSU_SP), rasstring[0], rasstring[1], rasstring[2], rasstring[3], iarstring
    );
    printf
    (   "                                         $%s $%s $%s $%s\n\n",
        rasstring[4], rasstring[5], rasstring[6], rasstring[7]
    );
        REACTIVATE;
}

// ** Update condition code reg **
MODULE __inline void set_cc(UBYTE n) // set CC
{   psl &= ~(PSL_CC); // clear CC bits of PSL

    if (n >= 128)
    {   psl |= 0x80; // CC = %10
    } elif (n > 0)
    {   psl |= 0x40; // CC = %01
}   }

// ** Push Return Address Stack **
MODULE __inline void pushras(void)
{   // store address
    ras[psu & PSU_SP] = (SLONG) page + iar + table_sz[opcode]; // wraparound issues?

    // increment SP
    if ((psu & PSU_SP) == 7) // overflow
    {   psu &= ~(PSU_SP); // set SP to 0
    } else
    {   psu++;
}   }

// ** Pull Return Address Stack **
MODULE __inline void pullras(void)
{   // decrement SP
    if ((psu & PSU_SP) == 0) // underflow
    {   psu |= PSU_SP; // set SP to 7
    } else psu--;

    // retrieve address
    ea   = ras[psu & PSU_SP];
    page = (UWORD) (ea & PAGE);
    iar  = (UWORD) (ea & PMSK);
}

MODULE __inline void branch(void)
{   /* page 0 is $0000..$1FFF
       page 1 is $2000..$3FFF
       page 2 is $4000..$5FFF
       page 3 is $6000..$7FFF
    ippaaaaa aaaaaaaa */

    page = (UWORD) (ea & PAGE);
    iar  = (UWORD) (ea & PMSK);
    if (operand & 0x80)
    {   cycles += 6;
}   }

MODULE void compare(UBYTE first, UBYTE second)
{   int d;

    psl &= 0x3F; // clear CC
    if (psl & PSL_COM) // unsigned compare
    {   d = (UBYTE) first - (UBYTE) second;
    } else
    {   d = (SBYTE) first - (SBYTE) second;
    }
    if (d < 0 )
    {   psl |= 0x80; // %10 = lt
    } elif (d > 0)
    {   psl |= 0x40; // %01 = gt
}   }

// Add source to destination
// Add (if WC==0) or add with carry (if WC==1)
MODULE UBYTE add(UBYTE dest, UBYTE source)
{   before = dest;

    if (source <= 127 && dest <= 127)
    {   ovf = 1;
    } elif (source >= 128 && dest >= 128)
    {   ovf = -1;
    } else ovf = 0;

    res = dest + source + ((psl >> 3) & psl & PSL_C); // Apparently.

    psl &= ~(PSL_C | PSL_OVF | PSL_IDC);
    if (res & 0x100)
    {   psl |= PSL_C;
    }
    dest = res & 0xFF;
    if ((dest & 15) < (before & 15))
    {   psl |= PSL_IDC;
    }

    psl = (psl & ~(PSL_OVF | PSL_CC)) | ccc[dest + (((dest ^ before) << 1) & 256)];

    if (ovf == 0)
    {   psl &= ~(PSL_OVF);
    } elif (ovf == 1)
    {   if (dest >= 128)
        {   psl |= PSL_OVF;
        } else
        {   psl &= ~(PSL_OVF);
    }   }
    elif (ovf == -1)
    {   if (dest <= 127)
        {   psl |= PSL_OVF;
        } else
        {   psl &= ~(PSL_OVF);
    }   }
    /* OVF is set if both numbers were positive (<= 127) and the answer is negative (>= 128).
               or if both numbers were negative (>= 128) and the answer is positive (<= 127).
    Otherwise, it is cleared. */

    // set_cc() not necessary

    return((UBYTE) res);
}

// Subtract source from destination
// Subtract with borrow if WC flag of PSL is set
MODULE UBYTE subtract(UBYTE dest, UBYTE source)
{   before = dest;

    if (source <= 127 && dest <= 127)
    {   ovf = 1;
    } elif (source >= 128 && dest >= 128)
    {   ovf = -1;
    } else ovf = 0;

    res = dest - source - ((psl >> 3) & (psl ^ PSL_C) & PSL_C);
    psl &= ~(PSL_C | PSL_OVF | PSL_IDC);
    if ((res & 0x100) == 0)
    {   psl |= PSL_C;
    }
    dest = res & 0xFF;
    if ((dest & 15) <= (before & 15))
    {   psl |= PSL_IDC;
    }

    psl = (psl & ~(PSL_OVF | PSL_CC)) | ccc[dest + (((dest ^ before) << 1) & 256)];

    if (ovf == 0)
    {   psl &= ~(PSL_OVF);
    } elif (ovf == 1)
    {   if (dest >= 128)
        {   psl |= PSL_OVF;
        } else
        {   psl &= ~(PSL_OVF);
    }   }
    elif (ovf == -1)
    {   if (dest <= 127)
        {   psl |= PSL_OVF;
        } else
        {   psl &= ~(PSL_OVF);
    }   }
    /* OVF is set if both numbers were positive (<= 127) and the answer is negative (>= 128).
               or if both numbers were negative (>= 128) and the answer is positive (<= 127).
    Otherwise, it is cleared. */

    // set_cc() not necessary

    return((UBYTE) res);
}

/* ABS_EA(), etc. don't use cpu_read(), because they simply calculate an
   address (ie. a number), they don't actually look at the contents of
   that address. */

// ** build effective address with absolute addressing (non-branch) **
// called by LODA, EORA, ANDA, IORA, SUBA, ADDA, STRA, COMA
// ABS_EA() uses GET_RR; the others don't!
MODULE void ABS_EA(void)
{   /* build effective address within current 8K page */

    operand = memory[page + iar + 1];
    ea = page + (((operand << 8) + memory[page + iar + 2]) & PMSK);
    GET_RR;

    if (operand & 0x80) /* indirect addressing? */
    {   addr = (int) ea;
        /* build indirect 32K address */
        ea = memory[addr] << 8;
        if ((++addr & PMSK) == 0)
        {   addr -= PLEN; /* page wrap */
        }
        ea = (ea + memory[addr]) & AMSK;
        SIX_CYCLES;
    }

    /* check indexed addressing modes */
    switch (operand & 0x60)
    {
    case 0x20: /* auto increment indexed */
        r[rr]++;
        ea = (ea & PAGE) + ((ea + r[rr]) & PMSK);
        rr = 0; /* absolute addressing reg is R0 */
    acase 0x40: /* auto decrement indexed */
        r[rr]--;
        ea = (ea & PAGE) + ((ea + r[rr]) & PMSK);
        rr = 0; /* absolute addressing reg is R0 */
    acase 0x60: /* indexed */
        ea = (ea & PAGE) + ((ea + r[rr]) & PMSK);
        rr = 0; /* absolute addressing reg is R0 */
    adefault: // eg. $00 (not indexed)
    break;
}   }

// ** build effective address with absolute addressing (branch) **
// called by BCTA, BRNA, BCFA, BIRA, BDRA, BXA, BSXA, BSTA, BSNA, BSFA
MODULE void BRA_EA(void)
{   /* build address in 32K address space */

    operand = memory[page + iar + 1];
    ea = ((operand << 8) + memory[page + iar + 2]) & AMSK;

    if (operand & 0x80) // indirect addressing?
    {   addr = (int) ea;
        /* build indirect 32K address */
        ea = memory[addr] << 8;
        if ((++addr & PMSK) == 0)
        {   addr -= PLEN; /* page wrap */
        }
        ea = (ea + memory[addr]) & AMSK;
        SIX_CYCLES;
}   }

// ** build effective address with relative addressing **
// called by LODR, BCTR, BSTR, EORR, ANDR, BRNR, IORR, BSNR, ADDR,
// BCFR, SUBR, BSFR, STRR, BIRR, COMR, BDRR
MODULE void REL_EA(void)
{   /* build effective address within current 8K page */

    operand = memory[page + iar + 1];
    ea = page + ((iar + 2 + S2650_relative[operand]) & PMSK);

    if (operand & 0x80) /* indirect bit set? */
    {   addr = (int) ea;
        /* build indirect 32K address */
        ea = memory[addr] << 8;
        if ((++addr & PMSK) == 0)
        {   addr -= PLEN; /* page wrap */
        }
        ea = (ea + memory[addr]) & AMSK;
        SIX_CYCLES;
}   }

// ** build effective address with zero relative addressing **
// called by ZBRR, ZBSR
MODULE void ZERO_EA(void)
{   /* build effective address from 0 */

    operand = memory[page + iar + 1];
    ea = (S2650_relative[operand] & PMSK);

    if (operand & 0x80) // indirect bit set?
    {   addr = (int) ea;
        /* build indirect 32K address */
        ea = memory[addr] << 8;
        if ((++addr & PMSK) == 0)
        {   addr -= PLEN; /* page wrap */
        }
        ea = (ea + memory[addr]) & AMSK;
        SIX_CYCLES;
}   }

MODULE void ABS_EA_TRACE(void)
{   ea = page + (((operand << 8) + memory[page + iar + 2]) & PMSK);

    // addressstring must be made BEFORE the main string
    if (operand & 0x80) /* indirect addressing? */
    {   addressstring[0] = '*';
        addressstring[1] = '$';
        hex2(&addressstring[2], (UWORD) ea);
    } else
    {   addressstring[0] = '$';
        hex2(&addressstring[1], (UWORD) ea);
    }

    /* check indexed addressing modes */
    switch (operand & 0x60)
    {
    case 0x20: /* auto increment indexed */
        strcat(addressstring, ",r");
        strcat(addressstring, reg);
        strcat(addressstring, "+");
        reg[0] = '0';
    acase 0x40: /* auto decrement indexed */
        strcat(addressstring, ",r");
        strcat(addressstring, reg);
        strcat(addressstring, "-");
        reg[0] = '0';
    acase 0x60: /* indexed */
        strcat(addressstring, ",r");
        strcat(addressstring, reg);
        reg[0] = '0';
    adefault: // non-indexed
    break;
}   }

MODULE void REL_EA_TRACE(void)
{   ea = page + ((iar + 2 + S2650_relative[operand]) & PMSK);

    if (operand & 0x80) /* indirect bit set? */
    {   addressstring[0] = '*';
        addressstring[1] = '$';
        hex2(&addressstring[2], (UWORD) ea);
    } else
    {   addressstring[0] = '$';
        hex2(&addressstring[1], (UWORD) ea);
}   }

MODULE void BRA_EA_TRACE(void)
{   ea = ((operand << 8) + memory[page + iar + 2]) & AMSK;

    if (operand & 0x80) // indirect addressing?
    {   addressstring[0] = '*';
        addressstring[1] = '$';
        hex2(&addressstring[2], (UWORD) ea);
    } else
    {   addressstring[0] = '$';
        hex2(&addressstring[1], (UWORD) ea);
}   }

MODULE void ZERO_EA_TRACE(void)
{   ea = (S2650_relative[operand] & PMSK);

    if (operand & 0x80) // indirect bit set?
    {   addressstring[0] = '*';
        addressstring[1] = '$';
        hex2(&addressstring[2], (UWORD) ea);
    } else
    {   addressstring[0] = '$';
        hex2(&addressstring[1], (UWORD) ea);
}   }

MODULE __inline void tracecpu(ULONG cycles)
{   operand = memory[page + iar + 1];

    GET_RR;
    reg[0] = '0' + rr;

    decimal(mn, cycles);
    mn[10] = ' ';
    hex2(&mn[11], (UWORD) (page + iar));
    mn[15] = ' ';

/* typical line is:
clock      iar  machine  assembler
1234567890 0000 00       EORZ    R0
01234567890123456789012345678901234
*/
    hex1(byte1, opcode);

    if (table_sz[opcode] == 3)
    {   hex1(byte2, operand);
        hex1(byte3, memory[page + iar + 2]);
        sprintf(&mn[16], "%s %s %s ", byte1, byte2, byte3);
    } elif (table_sz[opcode] == 2)
    {   hex1(byte2, operand);
        sprintf(&mn[16], "%s %s    ", byte1, byte2);
    } else
    {   // assert(table_sz[opcode] == 1);
        sprintf(&mn[16], "%s       ", byte1);
    }

    switch(opcode)
    {
    case 0:
    case 1:
    case 2:
    case 3:
        strcat(mn, "LODZ    r");
        strcat(mn, reg);
    acase 4:
    case 5:
    case 6:
    case 7:
        stcl_h(v, operand);
        strcat(mn, "LODI,r");
        strcat(mn, reg);
        strcat(mn, " $");
        strcat(mn, v);
    acase 8:
    case 9:
    case 10:
    case 11:
        REL_EA_TRACE();
        strcat(mn, "LODR,r");
        strcat(mn, reg);
        strcat(mn, " ");
        strcat(mn, addressstring);
    acase 12:
    case 13:
    case 14:
    case 15:
        ABS_EA_TRACE();
        strcat(mn, "LODA,r");
        strcat(mn, reg);
        strcat(mn, " ");
        strcat(mn, addressstring);
    acase 18:
        strcat(mn, "SPSU");
    acase 19:
        strcat(mn, "SPSL");
    acase 20:
    case 21:
    case 22:
    case 23:
        strcat(mn, "RETC,");
        strcat(mn, ccstring[BRANCHCODE]);
    acase 24:
    case 25:
    case 26:
    case 27:
        REL_EA_TRACE();
        strcat(mn, "BCTR,");
        strcat(mn, ccstring[BRANCHCODE]);
        strcat(mn, " ");
        strcat(mn, addressstring);
    acase 28:
    case 29:
    case 30:
    case 31:
        BRA_EA_TRACE();
        strcat(mn, "BCTA,");
        strcat(mn, ccstring[BRANCHCODE]);
        strcat(mn, " ");
        strcat(mn, addressstring);
    acase 32:
    case 33:
    case 34:
    case 35:
        strcat(mn, "EORZ    r");
        strcat(mn, reg);
    acase 36:
    case 37:
    case 38:
    case 39:
        stcl_h(v, operand);
        strcat(mn, "EORI,r");
        strcat(mn, reg);
        strcat(mn, " $");
        strcat(mn, v);
    acase 40:
    case 41:
    case 42:
    case 43:
        REL_EA_TRACE();
        strcat(mn, "EORR,r");
        strcat(mn, reg);
        strcat(mn, " ");
        strcat(mn, addressstring);
    acase 44:
    case 45:
    case 46:
    case 47:
        ABS_EA_TRACE();
        strcat(mn, "EORA,r");
        strcat(mn, reg);
        strcat(mn, " ");
        strcat(mn, addressstring);
    acase 52:
    case 53:
    case 54:
    case 55:
        strcat(mn, "RETE,");
        strcat(mn, ccstring[BRANCHCODE]);
    acase 56:
    case 57:
    case 58:
    case 59:
        REL_EA_TRACE();
        strcat(mn, "BSTR,");
        strcat(mn, ccstring[BRANCHCODE]);
        strcat(mn, " ");
        strcat(mn, addressstring);
    acase 60:
    case 61:
    case 62:
    case 63:
        BRA_EA_TRACE();
        strcat(mn, "BSTA,");
        strcat(mn, ccstring[BRANCHCODE]);
        strcat(mn, " ");
        strcat(mn, addressstring);
    acase 64:
        strcat(mn, "HALT");
    acase 65:
    case 66:
    case 67:
        strcat(mn, "ANDZ    r");
        strcat(mn, reg);
    acase 68:
    case 69:
    case 70:
    case 71:
        stcl_h(v, operand);
        strcat(mn, "ANDI,r");
        strcat(mn, reg);
        strcat(mn, " $");
        strcat(mn, v);
    acase 72:
    case 73:
    case 74:
    case 75:
        REL_EA_TRACE();
        strcat(mn, "ANDR,r");
        strcat(mn, reg);
        strcat(mn, " ");
        strcat(mn, addressstring);
    acase 76:
    case 77:
    case 78:
    case 79:
        ABS_EA_TRACE();
        strcat(mn, "ANDA,r");
        strcat(mn, reg);
        strcat(mn, " ");
        strcat(mn, addressstring);
    acase 80:
    case 81:
    case 82:
    case 83:
        strcat(mn, "RRR,r");
        strcat(mn, reg);
    acase 88:
    case 89:
    case 90:
    case 91:
        REL_EA_TRACE();
        strcat(mn, "BRNR,r");
        strcat(mn, reg);
        strcat(mn, " ");
        strcat(mn, addressstring);
    acase 92:
    case 93:
    case 94:
    case 95:
        BRA_EA_TRACE();
        strcat(mn, "BRNA,r");
        strcat(mn, reg);
        strcat(mn, " ");
        strcat(mn, addressstring);
    acase 96:
    case 97:
    case 98:
    case 99:
        strcat(mn, "IORZ    r");
        strcat(mn, reg);
    acase 100:
    case 101:
    case 102:
    case 103:
        stcl_h(v, operand);
        strcat(mn, "IORI,r");
        strcat(mn, reg);
        strcat(mn, " $");
        strcat(mn, v);
    acase 104:
    case 105:
    case 106:
    case 107:
        REL_EA_TRACE();
        strcat(mn, "IORR,r");
        strcat(mn, reg);
        strcat(mn, " ");
        strcat(mn, addressstring);
    acase 108:
    case 109:
    case 110:
    case 111:
        ABS_EA_TRACE();
        strcat(mn, "IORA,r");
        strcat(mn, reg);
        strcat(mn, " ");
        strcat(mn, addressstring);
    acase 116:
        stcl_h(v, operand);
        strcat(mn, "CPSU    $");
        strcat(mn, v);
    acase 117:
        stcl_h(v, operand);
        strcat(mn, "CPSL    $");
        strcat(mn, v);
    acase 118:
        stcl_h(v, operand);
        strcat(mn, "PPSU    $");
        strcat(mn, v);
    acase 119:
        stcl_h(v, operand);
        strcat(mn, "PPSL    $");
        strcat(mn, v);
    acase 120:
    case 121:
    case 122:
    case 123:
        REL_EA_TRACE();
        strcat(mn, "BSNR,r");
        strcat(mn, reg);
        strcat(mn, " ");
        strcat(mn, addressstring);
    acase 124:
    case 125:
    case 126:
    case 127:
        BRA_EA_TRACE();
        strcat(mn, "BSNA,r");
        strcat(mn, reg);
        strcat(mn, " ");
        strcat(mn, addressstring);
    acase 128:
    case 129:
    case 130:
    case 131:
        strcat(mn, "ADDZ    r");
        strcat(mn, reg);
    acase 132:
    case 133:
    case 134:
    case 135:
        stcl_h(v, operand);
        strcat(mn, "ADDI,r");
        strcat(mn, reg);
        strcat(mn, " $");
        strcat(mn, v);
    acase 136:
    case 137:
    case 138:
    case 139:
        REL_EA_TRACE();
        strcat(mn, "ADDR,r");
        strcat(mn, reg);
        strcat(mn, " ");
        strcat(mn, addressstring);
    acase 140:
    case 141:
    case 142:
    case 143:
        ABS_EA_TRACE();
        strcat(mn, "ADDA,r");
        strcat(mn, reg);
        strcat(mn, " ");
        strcat(mn, addressstring);
    acase 146:
        strcat(mn, "LPSU    r0");
    acase 147:
        strcat(mn, "LPSL    r0");
    acase 148:
    case 149:
    case 150:
    case 151:
        strcat(mn, "DAR     r");
        strcat(mn, reg);
    acase 152:
    case 153:
    case 154:
        REL_EA_TRACE();
        strcat(mn, "BCFR,");
        strcat(mn, ccstring[BRANCHCODE]);
        strcat(mn, " ");
        strcat(mn, addressstring);
    acase 155:
        ZERO_EA_TRACE();
        strcat(mn, "ZBRR,un ");
        strcat(mn, addressstring);
    acase 156:
    case 157:
    case 158:
        BRA_EA_TRACE();
        strcat(mn, "BCFA,");
        strcat(mn, ccstring[BRANCHCODE]);
        strcat(mn, " ");
        strcat(mn, addressstring);
    acase 159:
        BRA_EA_TRACE();
        if (psl & PSL_RS)
        {   strcat(mn, "BXA,r6  ");
        } else
        {   strcat(mn, "BXA,r3  ");
        }
        strcat(mn, addressstring);
    acase 160:
    case 161:
    case 162:
    case 163:
        strcat(mn, "SUBZ    r");
        strcat(mn, reg);
    acase 164:
    case 165:
    case 166:
    case 167:
        stcl_h(v, operand);
        strcat(mn, "SUBI,r");
        strcat(mn, reg);
        strcat(mn, " $");
        strcat(mn, v);
    acase 168:
    case 169:
    case 170:
    case 171:
        REL_EA_TRACE();
        strcat(mn, "SUBR,r");
        strcat(mn, reg);
        strcat(mn, " ");
        strcat(mn, addressstring);
    acase 172:
    case 173:
    case 174:
    case 175:
        ABS_EA_TRACE();
        strcat(mn, "SUBA,r");
        strcat(mn, reg);
        strcat(mn, " ");
        strcat(mn, addressstring);
    acase 180:
        stcl_h(v, operand);
        strcat(mn, "TPSU    $");
        strcat(mn, v);
    acase 181:
        stcl_h(v, operand);
        strcat(mn, "TPSL    $");
        strcat(mn, v);
    acase 184:
    case 185:
    case 186:
        REL_EA_TRACE();
        strcat(mn, "BSFR,");
        strcat(mn, ccstring[BRANCHCODE]);
        strcat(mn, " ");
        strcat(mn, addressstring);
    acase 187:
        ZERO_EA_TRACE();
        strcat(mn, "ZBSR,un ");
        strcat(mn, addressstring);
    acase 188:
    case 189:
    case 190:
        BRA_EA_TRACE();
        strcat(mn, "BSFA,");
        strcat(mn, ccstring[BRANCHCODE]);
        strcat(mn, " ");
        strcat(mn, addressstring);
    acase 191:
        BRA_EA_TRACE();
        if (psl & PSL_RS)
        {   strcat(mn, "BSXA,r6 ");
        } else
        {   strcat(mn, "BSXA,r3 ");
        }
        strcat(mn, addressstring);
    acase 192:
        strcat(mn, "NOP");
    acase 193:
    case 194:
    case 195:
        strcat(mn, "STRZ    r");
        strcat(mn, reg);
    acase 200:
    case 201:
    case 202:
    case 203:
        REL_EA_TRACE();
        strcat(mn, "STRR,r");
        strcat(mn, reg);
        strcat(mn, " ");
        strcat(mn, addressstring);
    acase 204:
    case 205:
    case 206:
    case 207:
        ABS_EA_TRACE();
        strcat(mn, "STRA,r");
        strcat(mn, reg);
        strcat(mn, " ");
        strcat(mn, addressstring);
    acase 208:
    case 209:
    case 210:
    case 211:
        strcat(mn, "RRL,r");
        strcat(mn, reg);
    acase 216:
    case 217:
    case 218:
    case 219:
        REL_EA_TRACE();
        strcat(mn, "BIRR,r");
        strcat(mn, reg);
        strcat(mn, " ");
        strcat(mn, addressstring);
    acase 220:
    case 221:
    case 222:
    case 223:
        BRA_EA_TRACE();
        strcat(mn, "BIRA,r");
        strcat(mn, reg);
        strcat(mn, " ");
        strcat(mn, addressstring);
    acase 224:
    case 225:
    case 226:
    case 227:
        strcat(mn, "COMZ    r");
        strcat(mn, reg);
    acase 228:
    case 229:
    case 230:
    case 231:
        stcl_h(v, operand);
        strcat(mn, "COMI,r");
        strcat(mn, reg);
        strcat(mn, " $");
        strcat(mn, v);
    acase 232:
    case 233:
    case 234:
    case 235:
        REL_EA_TRACE();
        strcat(mn, "COMR,r");
        strcat(mn, reg);
        strcat(mn, " ");
        strcat(mn, addressstring);
    acase 236:
    case 237:
    case 238:
    case 239:
        ABS_EA_TRACE();
        strcat(mn, "COMA,r");
        strcat(mn, reg);
        strcat(mn, " ");
        strcat(mn, addressstring);
    acase 244:
    case 245:
    case 246:
    case 247:
        stcl_h(v, operand);
        strcat(mn, "TMI,r");
        strcat(mn, reg);
        strcat(mn, "  $");
        strcat(mn, v);
    acase 248:
    case 249:
    case 250:
    case 251:
        REL_EA_TRACE();
        strcat(mn, "BDRR,r");
        strcat(mn, reg);
        strcat(mn, " ");
        strcat(mn, addressstring);
    acase 252:
    case 253:
    case 254:
    case 255:
        BRA_EA_TRACE();
        strcat(mn, "BDRA,r");
        strcat(mn, reg);
        strcat(mn, " ");
        strcat(mn, addressstring);
    // useless opcodes----------------------------------------------------
    acase 48:
    case 49:
    case 50:
    case 51:
        strcat(mn, "REDC r");
        strcat(mn, reg);
    acase 84:
    case 85:
    case 86:
    case 87:
        stcl_h(v, operand);
        strcat(mn, "REDE,r");
        strcat(mn, reg);
        strcat(mn, " $");
        strcat(mn, v);
    acase 112:
    case 113:
    case 114:
    case 115:
        strcat(mn, "REDD    r");
        strcat(mn, reg);
    acase 176:
    case 177:
    case 178:
    case 179:
        strcat(mn, "WRTC    r");
        strcat(mn, reg);
    acase 212:
    case 213:
    case 214:
    case 215:
        stcl_h(v, operand);
        strcat(mn, "WRTE,r");
        strcat(mn, reg);
        strcat(mn, " $");
        strcat(mn, v);
    acase 240:
    case 241:
    case 242:
    case 243:
        strcat(mn, "WRTD    r");
        strcat(mn, reg);
    adefault:
        strcat(mn, "?");
    break;
}   }
